/**
* \file: IControlAdapter.h
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: CarPlay
*
* \author: J. Harder / ADIT/SW1 / jharder@de.adit-jv.com
*
* \copyright (c) 2013-2014 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#ifndef CARPLAY_ICONTROLADAPTER_H
#define CARPLAY_ICONTROLADAPTER_H

#include "IConfiguration.h"
#include "IAudioOutAdapter.h"

#include <netinet/in.h>

namespace adit { namespace carplay
{

typedef enum
{
    Entity_NA     = 0,
    Entity_Car    = 1,
    Entity_Mobile = 2,
} __attribute__ ((packed, aligned (1))) Entity;

typedef enum
{
    SpeechMode_NA          = 0,
    SpeechMode_None        = 1,
    SpeechMode_Recognizing = 2,
    SpeechMode_Speaking    = 3
} __attribute__ ((packed, aligned (1))) SpeechMode;

struct SpeechState
{
    Entity     entity /*= Entity_NA*/;
    SpeechMode mode   /*= SpeechMode_NA*/;

    SpeechState();
} __attribute__ ((packed));

struct ModeState
{
    Entity      screen     /*= Entity_NA*/;
    Entity      audio      /*= Entity_NA*/;
    SpeechState speech;
    Entity      phone      /*= Entity_NA*/;
    Entity      navigation /*= Entity_NA*/;

    ModeState();
};

typedef enum
{
    TransferType_NA       = 0,
    TransferType_Take     = 1,
    TransferType_Untake   = 2,
    TransferType_Borrow   = 3,
    TransferType_Unborrow = 4
} __attribute__ ((packed, aligned (1))) TransferType;

typedef enum
{
    TransferPriority_NA            = 0,
    TransferPriority_NiceToHave    = 1,
    TransferPriority_UserInitiated = 2
} __attribute__ ((packed, aligned (1))) TransferPriority;

typedef enum
{
    Constraint_NA            = 0,
    Constraint_Anytime       = 1,
    Constraint_UserInitiated = 2,
    Constraint_Never         = 3
} __attribute__ ((packed, aligned (1))) Constraint;

typedef enum
{
    AppState_NA    = 0,
    AppState_True  = 1,
    AppState_False = 2
} __attribute__ ((packed, aligned (1))) AppState;

struct ResourceTransfer
{
    TransferType     type                       /*= TransferType_NA*/;
    TransferPriority priority                   /*= TransferPriority_NA*/;
    Constraint       takeConstraint             /*= Constraint_NA*/;
    Constraint       borrowOrUnborrowConstraint /*= Constraint_NA*/;

    ResourceTransfer();
} __attribute__ ((packed, aligned (4)));

struct ModeChanges
{
    ResourceTransfer screen;
    ResourceTransfer audio;
    SpeechMode       speech     /*= SpeechMode_NA*/;
    AppState         phone      /*= AppState_NA*/;
    AppState         navigation /*= AppState_NA*/;

    ModeChanges();
};

typedef enum
{
    ETCInfo_NotSupported   = -1,
    ETCInfo_InActive       = 0,
    ETCInfo_Active         = 1
} ETCInfo;

typedef enum
{
    NADInfo_NotSupported   = -1,
    NADInfo_InActive       = 0,
    NADInfo_Active         = 1
} NADInfo;

struct VehicleInformation
{
    ETCInfo electronicTollSupport;
    NADInfo navigationAidedDrivSupport;
};

typedef enum
{
    SiriAction_NA           = 0,
    SiriAction_Prewarm      = 1,
    SiriAction_ButtonDown   = 2,
    SiriAction_ButtonUp     = 3
} SiriAction;

typedef enum
{
    VersionEntity_CarPlay         = 0,     /*= CarPlay version */
    VersionEntity_ApplePluginCore = 1,     /*= Apple delivered communication plug-in code version  */
    VersionEntity_PhoneiOS        = 2      /*= iPhone iOS version */
} VersionEntity;

typedef enum
{
    CarPlay_Over_USB		= 0,
    CarPlay_Over_WiFi		= 1
}CarPlayTransportType;

/**
 * Interface provided by CarPlay.
 */
class IControlReceiver
{
public:
    virtual ~IControlReceiver() {};

    /**
     * Request a change of resource ownership or application state.
     * Respond is sent via OnModesChanged on IControlAdapter.
     *
     * \param [in] inChanges    requested changes.
     *
     * \see Apple specification.
     */
    virtual void ChangeResourceMode(const ModeChanges& inChanges) = 0;

    /**
     * Request the UI of the iOS device to show.
     *
     * \param [in] inUrl        URL identify of the application to launch
     *
     * \see Apple specification.
     */
    virtual void RequestUI(const std::string& inUrl) = 0;

    /**
     * Request a Siri action.
     *
     * \param [in] inAction     Siri action
     *
     * \see Apple specification.
     */
    virtual void RequestSiriAction(SiriAction inAction) = 0;

    /**
     * Set night mode.
     *
     * \param [in] inActive     true: night mode is active
     *
     * \see Apple specification.
     */
    virtual void SetNightMode(bool inActive) = 0;

    /**
     * Set limited UI mode.
     *
     * \param [in] inActive     true: limited UI is active
     *
     * \see Apple specification.
     */
    virtual void SetLimitedUI(bool inActive) = 0;

    /**
     * Set Bluetooth IDs of Bluetooth modules.
     */
    virtual void SetBluetoothIDs(const std::list<std::string>& inDeviceIDs) = 0;

    /**
     *  Get Version.
     *
     * \param [in] inEntity           Enum value of requested  version information
     *                                (CarPlay , ApplePluginCore and connected iPhone iOS version)
     * \return string                 string reference consist of version information
     * \.
     */
    virtual const std::string& GetVersion(VersionEntity inEntity) = 0;

    /**
     *  Get IP Address.
     *
     * \return struct sockaddr const * Can be casted to struct sockaddr_in const * for IPv4 connections
     *                                 or struct sockaddr_in6 const * for IPv6 connections
     * \.
     */
    virtual struct sockaddr const * getIPAddress(void) = 0;

    /**
     * Update state of Vehicle Information.
     *
     * \param [in] Vehicle Information
     *
     * \see Apple specification.
     */
    virtual void UpdateVehicleInformation(VehicleInformation inVehicleInfo) = 0;
};

/**
 * Interface to be implemented by the adapter.
 */
class IControlAdapter
{
public:
    /**
     * Default virtual destructor.
     * Gets called when a Session is destroyed.
     */
    virtual ~IControlAdapter() {};

    /**
     * Initialize control adapter.
     * Gets called when a new Session is created.
     */
    virtual bool Initialize(IDynamicConfiguration& inConfig, IControlReceiver& inControl) = 0;

    /**
     * Notification of changed resource ownership or application state.
     *
     * \param [in] inState       structure containing current state machine
     *
     * \see Apple specification.
     */
    virtual void OnModesChanged(ModeState inState) = 0;

    /**
     * deprecated: don't use in new code
     *
     * Notification of audio stream in preparation.
     *
     * \param [in] inChannel     channel type: main or alternate
     * \param [in] inAudioType   type of audio
     *
     * \see Apple specification for audio types.
     */
    virtual void OnAudioPrepare(AudioChannelType inChannel, const std::string& inAudioType) = 0;

    /**
     * deprecated: don't use in new code
     */
    virtual void OnAudioStop(AudioChannelType inChannel) = 0;

    /**
     *
     * Notification of audio stream flush.
     *
     * \see Apple specification for audio types.
	 * */

    //virtual void OnAudioFlush() = 0;
    /**
     * Request from the iOS device to display native HMI.
     *
     *\param [in] inURL URL of the desired native UI to show
     *
     * \see Apple specification.
     */
    virtual void OnRequestUI(const std::string& inURL) = 0;

    /**
     * Ramp audio volume up or down.
     *
     * \param [in] inFinalVolume final volume
     * \param [in] inDuration    duration in seconds
     *
     * \see Apple specification.
     */
    virtual void OnRampVolume(double inFinalVolume, double inDuration) = 0;

    /**
     * Request from iOS device to disable a specified Bluetooth device.
     *
     * \param [in] inDeviceID   MAC address of the Bluetooth device
     *
     * \see Apple specification.
     */
    virtual void OnDisableBluetooth(const std::string& inDeviceID) = 0;

    /*
     * deprecated: don't use in new code
     */
    virtual void OnGetBluetoothIDs(std::list<std::string>& outDeviceIDs) = 0;

    /*
     * deprecated: don't use in new code
     */
    virtual bool OnGetNightMode() = 0;

    /*
     * Request from iOS device to get currently issued resource mode changes.
     * This method is used when constructing the INFO message for initial setup, but it may be
     * also called at any later time.
     *
     * \param [out] outChanges  Currently issued resource mode changes.
     */
    virtual void OnGetCurrentResourceMode(ModeChanges& outChanges) { outChanges = ModeChanges(); };

    /*
     * Notification that phone and stack are ready.
     * ChangeResourceMode, RequestUI are now accepted.
     */
    virtual void OnSessionStart(CarPlayTransportType inTransportType) = 0;

    /*
     * Notification that the session is ended.
     * ChangeResourceMode, RequestUI are now not accepted anymore.
     */
    virtual void OnSessionEnd() { };

    /*
     * Notification that video playback is started.
     */
    virtual void OnVideoPlaybackStarted() { };
};

} } /* namespace adit { namespace carplay */

#endif /* CARPLAY_ICONTROLADAPTER_H */
